
# -*- coding: utf-8 -*-
"""
Telegram Academy Bot — Single File
- Free/Paid registration
- Zarinpal payment (request/verify)
- 15-question quiz (1-5 scale) with analysis and buy button
- SMS (IPPanel) + Voice (Avanak)
- Admin panel: /manage, /export, /broadcast, /set_text
- SQLite storage
"""
import os, json, sqlite3, time, csv, io, requests, datetime, random
from telebot import TeleBot, types

# ---------------- Config & Content ----------------
CONTENT_PATH = "content.json"
DB_PATH = "registrations.db"
BOT_TOKEN = os.environ.get("BOT_TOKEN", "000:TEST")

ADMIN_IDS = set(int(x) for x in os.environ.get("ADMIN_IDS", "212732920").split(",") if x.strip())

def load_content():
    with open(CONTENT_PATH, "r", encoding="utf-8") as f:
        return json.load(f)

def save_content(c):
    with open(CONTENT_PATH, "w", encoding="utf-8") as f:
        json.dump(c, f, ensure_ascii=False, indent=2)

C = load_content()
bot = TeleBot(BOT_TOKEN, parse_mode="HTML")

# ---------------- Database ----------------
def db():
    conn = sqlite3.connect(DB_PATH, check_same_thread=False)
    conn.row_factory = sqlite3.Row
    return conn

def init_db():
    conn = db()
    cur = conn.cursor()
    cur.execute("""CREATE TABLE IF NOT EXISTS users(
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        tg_id INTEGER UNIQUE,
        full_name TEXT,
        phone TEXT,
        created_at TEXT
    )""")
    cur.execute("""CREATE TABLE IF NOT EXISTS courses(
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        title TEXT,
        is_free INTEGER,
        price INTEGER
    )""")
    cur.execute("""CREATE TABLE IF NOT EXISTS registrations(
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        user_id INTEGER,
        course_id INTEGER,
        paid INTEGER,
        payment_ref TEXT,
        created_at TEXT
    )""")
    cur.execute("""CREATE TABLE IF NOT EXISTS payments(
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        user_id INTEGER,
        course_id INTEGER,
        authority TEXT,
        amount INTEGER,
        status TEXT,
        created_at TEXT
    )""")
    cur.execute("""CREATE TABLE IF NOT EXISTS quiz_attempts(
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        user_id INTEGER,
        score INTEGER,
        created_at TEXT
    )""")
    conn.commit()
    # seed courses
    cur.execute("SELECT COUNT(*) as c FROM courses")
    if cur.fetchone()["c"] == 0:
        cur.execute("INSERT INTO courses(title,is_free,price) VALUES(?,?,?)", ("وبینار رایگان فن بیان", 1, 0))
        cur.execute("INSERT INTO courses(title,is_free,price) VALUES(?,?,?)", ("دوره جامع فن بیان", 0, 9900000))
        conn.commit()
    conn.close()

init_db()

# ---------------- State (simple) ----------------
user_state = {}  # tg_id -> dict (stage, data)
admin_waiting_key = {}  # tg_id -> 'content.key'

# ---------------- Helpers ----------------
def main_menu():
    kb = types.ReplyKeyboardMarkup(resize_keyboard=True)
    kb.add(types.KeyboardButton(C["menus"]["free"]))
    kb.add(types.KeyboardButton(C["menus"]["paid"]))
    kb.add(types.KeyboardButton(C["menus"]["quiz"]))
    kb.add(types.KeyboardButton(C["menus"]["admin"]))
    return kb

def ask_contact_kb():
    kb = types.ReplyKeyboardMarkup(resize_keyboard=True)
    btn = types.KeyboardButton("📱 ارسال شماره", request_contact=True)
    kb.add(btn)
    return kb

def get_or_create_user(tg_id, full_name):
    conn = db(); cur = conn.cursor()
    cur.execute("SELECT * FROM users WHERE tg_id=?", (tg_id,))
    row = cur.fetchone()
    if row: 
        conn.close(); return row
    cur.execute("INSERT INTO users(tg_id,full_name,phone,created_at) VALUES(?,?,?,?)",
                (tg_id, full_name or "", None, datetime.datetime.utcnow().isoformat()))
    conn.commit()
    cur.execute("SELECT * FROM users WHERE tg_id=?", (tg_id,))
    row = cur.fetchone()
    conn.close()
    return row

def send_sms_ippanel(phone, text):
    # Adjust to your IPPanel API shape
    api_key = C["integrations"]["ippanel"]["api_key"]
    sender = C["integrations"]["ippanel"]["sender"]
    try:
        url = "https://ippanel.com/api/select"
        payload = {
            "op": "send",
            "uname": api_key,
            "message": text,
            "from": sender,
            "to": [phone]
        }
        r = requests.post(url, json=payload, timeout=15)
        return r.status_code, r.text
    except Exception as e:
        return -1, str(e)

def call_avanak(phone, text=None, vars=None):
    api_key = C["integrations"]["avanak"]["api_key"]
    sender = C["integrations"]["avanak"]["sender"]
    scenario_id = C["integrations"]["avanak"]["scenario_id"]
    headers = {"Authorization": "Bearer " + api_key}
    try:
        if text:
            # TTS sample
            r = requests.post("https://api.avanak.ir/v1/tts/send", json={
                "to": phone, "from": sender, "text": text
            }, headers=headers, timeout=15)
            return r.status_code, r.text
        else:
            # Scenario
            r = requests.post("https://api.avanak.ir/v1/scenario/run", json={
                "to": phone, "from": sender, "scenario_id": scenario_id, "variables": vars or {}
            }, headers=headers, timeout=15)
            return r.status_code, r.text
    except Exception as e:
        return -1, str(e)

# ---------------- Payments (Zarinpal) ----------------
def zarinpal_request(amount_rials, description, phone, email="", callback_slug=""):
    merchant = C["integrations"]["zarinpal"]["merchant_id"]
    sandbox = C["integrations"]["zarinpal"]["sandbox"]
    base = "https://sandbox.zarinpal.com/pg/v4/payment" if sandbox else "https://api.zarinpal.com/pg/v4/payment"
    callback_base = C["integrations"]["zarinpal"]["callback_base"]
    secret = C["integrations"]["zarinpal"]["callback_secret"]
    callback_url = f"{callback_base}/zpcallback?slug={callback_slug}&secret={secret}"
    data = {
        "merchant_id": merchant,
        "amount": amount_rials,
        "description": description,
        "callback_url": callback_url,
        "metadata": {"mobile": phone, "email": email}
    }
    r = requests.post(base + "/request.json", json=data, timeout=15)
    try:
        js = r.json()
    except Exception:
        return None, f"bad response: {r.text}"
    if js.get("data") and js["data"].get("authority"):
        authority = js["data"]["authority"]
        startpay = ("https://sandbox.zarinpal.com/pg/StartPay/" if sandbox else "https://www.zarinpal.com/pg/StartPay/") + authority
        return {"authority": authority, "startpay": startpay}, None
    return None, js.get("errors", js)

def zarinpal_verify(authority, amount_rials):
    merchant = C["integrations"]["zarinpal"]["merchant_id"]
    sandbox = C["integrations"]["zarinpal"]["sandbox"]
    base = "https://sandbox.zarinpal.com/pg/v4/payment" if sandbox else "https://api.zarinpal.com/pg/v4/payment"
    data = {"merchant_id": merchant, "amount": amount_rials, "authority": authority}
    r = requests.post(base + "/verify.json", json=data, timeout=15)
    try:
        return r.json()
    except Exception:
        return {"error": r.text}

# ---------------- Handlers ----------------
@bot.message_handler(commands=["start"])
def on_start(m):
    get_or_create_user(m.from_user.id, m.from_user.full_name)
    bot.send_message(m.chat.id, C["prompts"]["welcome"], reply_markup=main_menu())

@bot.message_handler(func=lambda msg: msg.text == C["menus"]["free"])
def on_free(m):
    u = get_or_create_user(m.from_user.id, m.from_user.full_name)
    user_state[m.from_user.id] = {"stage": "ask_name_free", "data": {}}
    bot.send_message(m.chat.id, C["prompts"]["ask_name"])

@bot.message_handler(func=lambda msg: msg.text == C["menus"]["paid"])
def on_paid(m):
    # list paid courses
    conn = db(); cur = conn.cursor()
    cur.execute("SELECT title, price FROM courses WHERE is_free=0")
    rows = cur.fetchall(); conn.close()
    if not rows:
        bot.send_message(m.chat.id, "دوره پولی ثبت نشده.")
        return
    text = C["prompts"]["paid_choose"] + "\n\n" + "\n".join([f"• {r['title']} — {r['price']:,} تومان" for r in rows])
    bot.send_message(m.chat.id, text)
    user_state[m.from_user.id] = {"stage": "choose_paid", "data": {}}

@bot.message_handler(func=lambda msg: msg.text == C["menus"]["quiz"])
def on_quiz(m):
    user_state[m.from_user.id] = {"stage": "quiz", "q": 0, "answers": []}
    bot.send_message(m.chat.id, C["prompts"]["quiz_intro"])
    send_next_question(m.chat.id, m.from_user.id)

def send_next_question(chat_id, uid):
    st = user_state.get(uid)
    q_idx = st["q"]
    questions = C["quiz"]["questions"]
    if q_idx >= len(questions):
        finalize_quiz(chat_id, uid)
        return
    kb = types.ReplyKeyboardMarkup(resize_keyboard=True, row_width=5)
    kb.add(*[types.KeyboardButton(str(i)) for i in range(1,6)])
    bot.send_message(chat_id, questions[q_idx], reply_markup=kb)

def finalize_quiz(chat_id, uid):
    st = user_state.get(uid); answers = st.get("answers", [])
    total = sum(answers)
    # save
    conn = db(); cur = conn.cursor()
    cur.execute("SELECT id FROM users WHERE tg_id=?", (uid,)); urow = cur.fetchone()
    if urow:
        cur.execute("INSERT INTO quiz_attempts(user_id,score,created_at) VALUES(?,?,?)",
                    (urow["id"], total, datetime.datetime.utcnow().isoformat()))
        conn.commit()
    conn.close()
    # bucket
    low_max = C["quiz"]["buckets"]["low_max"]
    mid_max = C["quiz"]["buckets"]["mid_max"]
    if total <= low_max:
        analysis = C["quiz"]["analysis"]["low"]
    elif total <= mid_max:
        analysis = C["quiz"]["analysis"]["mid"]
    else:
        analysis = C["quiz"]["analysis"]["high"]
    # buy button
    kb = types.InlineKeyboardMarkup()
    kb.add(types.InlineKeyboardButton(C["quiz"]["buy_button_text"], callback_data="buy_paid"))
    bot.send_message(chat_id, f"نتیجه شما: <b>{total}</b> از ۷۵\n\n{analysis}", reply_markup=kb)
    # sms after quiz if phone exists
    conn = db(); cur = conn.cursor()
    cur.execute("SELECT phone FROM users WHERE tg_id=?", (uid,)); row = cur.fetchone()
    if row and row["phone"]:
        try: send_sms_ippanel(row["phone"], C["sms"]["after_quiz"])
        except: pass
    # clear state
    user_state.pop(uid, None)
    # restore main menu
    bot.send_message(chat_id, "منو:", reply_markup=main_menu())

@bot.callback_query_handler(func=lambda c: c.data=="buy_paid")
def on_buy_paid(c):
    # pick main paid course
    conn = db(); cur = conn.cursor()
    cur.execute("SELECT id,title,price FROM courses WHERE is_free=0 LIMIT 1")
    course = cur.fetchone(); conn.close()
    if not course:
        bot.answer_callback_query(c.id, "دوره پولی یافت نشد.")
        return
    # create payment
    # first need user phone
    conn = db(); cur = conn.cursor()
    cur.execute("SELECT phone,full_name,id FROM users WHERE tg_id=?", (c.from_user.id,))
    u = cur.fetchone(); conn.close()
    if not u or not u["phone"]:
        # ask phone then continue
        user_state[c.from_user.id] = {"stage": "need_phone_for_payment", "data": {"course_id": course["id"]}}
        bot.send_message(c.message.chat.id, "برای صدور لینک پرداخت، شماره‌ات را ارسال کن:", reply_markup=ask_contact_kb())
        bot.answer_callback_query(c.id)
        return
    start_payment(c.message.chat.id, c.from_user.id, course["id"], u["full_name"], u["phone"], course["price"])

def start_payment(chat_id, uid, course_id, name, phone, price_toman):
    amount_rials = int(price_toman)*10
    info, err = zarinpal_request(amount_rials, "خرید دوره آکادمی", phone, "", "order")
    if err or not info:
        bot.send_message(chat_id, "خطا در ایجاد لینک پرداخت.")
        return
    # save payment
    conn = db(); cur = conn.cursor()
    cur.execute("SELECT id FROM users WHERE tg_id=?", (uid,)); urow = cur.fetchone()
    cur.execute("INSERT INTO payments(user_id,course_id,authority,amount,status,created_at) VALUES(?,?,?,?,?,?)",
                (urow["id"], course_id, info["authority"], amount_rials, "created", datetime.datetime.utcnow().isoformat()))
    conn.commit(); conn.close()
    bot.send_message(chat_id, C["prompts"]["paid_pay_link"].format(pay_url=info["startpay"]))

@bot.message_handler(content_types=["contact"])
def on_contact(m):
    ph = m.contact.phone_number if m.contact else None
    if not ph:
        return
    u = get_or_create_user(m.from_user.id, m.from_user.full_name)
    conn = db(); cur = conn.cursor()
    cur.execute("UPDATE users SET phone=? WHERE tg_id=?", (ph, m.from_user.id))
    conn.commit(); conn.close()
    # check if waiting for payment phone
    st = user_state.get(m.from_user.id)
    if st and st.get("stage")=="need_phone_for_payment":
        course_id = st["data"]["course_id"]
        # fetch course
        conn = db(); cur = conn.cursor()
        cur.execute("SELECT title,price FROM courses WHERE id=?", (course_id,))
        course = cur.fetchone(); conn.close()
        start_payment(m.chat.id, m.from_user.id, course_id, u["full_name"], ph, course["price"])
        user_state.pop(m.from_user.id, None)
        return
    # normal free registration phone collection
    bot.send_message(m.chat.id, "شماره‌ات ثبت شد ✅")

@bot.message_handler(func=lambda msg: user_state.get(msg.from_user.id,{}).get("stage")=="ask_name_free")
def on_name_free(m):
    st = user_state[m.from_user.id]
    st["data"]["full_name"] = m.text.strip()
    user_state[m.from_user.id]["stage"] = "ask_phone_free"
    bot.send_message(m.chat.id, C["prompts"]["ask_phone"], reply_markup=ask_contact_kb())

@bot.message_handler(func=lambda msg: user_state.get(msg.from_user.id,{}).get("stage")=="choose_paid")
def on_choose_paid(m):
    title = m.text.strip()
    conn = db(); cur = conn.cursor()
    cur.execute("SELECT id,title,price FROM courses WHERE title=? AND is_free=0", (title,))
    row = cur.fetchone(); conn.close()
    if not row:
        bot.send_message(m.chat.id, C["prompts"]["paid_not_found"])
        return
    # stash chosen course
    user_state[m.from_user.id] = {"stage":"need_phone_for_payment", "data":{"course_id": row["id"]}}
    bot.send_message(m.chat.id, "برای صدور لینک پرداخت، شماره‌ات را بفرست:", reply_markup=ask_contact_kb())

@bot.message_handler(func=lambda msg: user_state.get(msg.from_user.id,{}).get("stage")=="quiz")
def on_quiz_answer(m):
    if m.text not in ["1","2","3","4","5"]:
        bot.send_message(m.chat.id, "لطفاً عدد ۱ تا ۵ را بفرست.")
        return
    st = user_state[m.from_user.id]
    st["answers"].append(int(m.text))
    st["q"] += 1
    send_next_question(m.chat.id, m.from_user.id)

# ---------------- Admin ----------------
@bot.message_handler(commands=["manage"])
def on_manage(m):
    if m.from_user.id not in ADMIN_IDS:
        return
    txt = (
        "🛠 پنل مدیریت:\n"
        "- /export_regs : خروجی CSV ثبت‌نام‌ها\n"
        "- /export_quiz : خروجی CSV نتایج آزمون\n"
        "- /broadcast متن : برودکست فوری\n"
        "- /set_text key : تغییر متن (بعدی مقدار)\n"
        "  کلیدهای پرکاربرد: prompts.welcome, sms.after_quiz, quiz.analysis.low\n"
        "- /set_price عنوان_دوره مبلغ_تومان\n"
    )
    bot.send_message(m.chat.id, txt)

@bot.message_handler(commands=["export_regs"])
def export_regs(m):
    if m.from_user.id not in ADMIN_IDS: return
    conn = db(); cur = conn.cursor()
    cur.execute("""SELECT u.full_name, u.phone, u.tg_id, c.title, r.paid, r.payment_ref, r.created_at
                   FROM registrations r
                   JOIN users u ON u.id=r.user_id
                   JOIN courses c ON c.id=r.course_id""")
    rows = cur.fetchall(); conn.close()
    output = io.StringIO()
    w = csv.writer(output)
    w.writerow(["full_name","phone","tg_id","course","paid","payment_ref","created_at"])
    for r in rows:
        w.writerow([r["full_name"], r["phone"], r["tg_id"], r["title"], r["paid"], r["payment_ref"], r["created_at"]])
    output.seek(0)
    bot.send_document(m.chat.id, ("registrations.csv", output.read()))

@bot.message_handler(commands=["export_quiz"])
def export_quiz(m):
    if m.from_user.id not in ADMIN_IDS: return
    conn = db(); cur = conn.cursor()
    cur.execute("""SELECT u.full_name, u.phone, q.score, q.created_at
                   FROM quiz_attempts q JOIN users u ON u.id=q.user_id""")
    rows = cur.fetchall(); conn.close()
    output = io.StringIO()
    w = csv.writer(output)
    w.writerow(["full_name","phone","score","created_at"])
    for r in rows:
        w.writerow([r["full_name"], r["phone"], r["score"], r["created_at"]])
    output.seek(0)
    bot.send_document(m.chat.id, ("quiz_starters.csv", output.read()))

@bot.message_handler(commands=["broadcast"])
def broadcast(m):
    if m.from_user.id not in ADMIN_IDS: return
    text = m.text.partition(" ")[2].strip()
    if not text:
        bot.reply_to(m, "بعد از دستور، متن را بنویس: /broadcast متن")
        return
    conn = db(); cur = conn.cursor()
    cur.execute("SELECT tg_id FROM users")
    tg_ids = [r["tg_id"] for r in cur.fetchall()]
    conn.close()
    sent = 0
    for uid in tg_ids:
        try:
            bot.send_message(uid, text)
            sent += 1
        except:
            pass
        time.sleep(0.05)
    bot.reply_to(m, f"ارسال شد: {sent} نفر")

@bot.message_handler(commands=["set_text"])
def set_text_key(m):
    if m.from_user.id not in ADMIN_IDS: return
    key = m.text.partition(" ")[2].strip()
    if not key:
        bot.reply_to(m, "کلید متن را بده: /set_text prompts.welcome")
        return
    admin_waiting_key[m.from_user.id] = key
    bot.reply_to(m, f"مقدار جدید برای «{key}» را ارسال کن.")

@bot.message_handler(func=lambda msg: msg.from_user.id in ADMIN_IDS and msg.from_user.id in admin_waiting_key)
def set_text_value(m):
    key = admin_waiting_key.pop(m.from_user.id)
    # set in C and save
    try:
        # navigate nested
        parts = key.split(".")
        ref = C
        for p in parts[:-1]:
            ref = ref[p]
        ref[parts[-1]] = m.text
        save_content(C)
        bot.reply_to(m, "ذخیره شد ✅")
    except Exception as e:
        bot.reply_to(m, f"خطا در ذخیره: {e}")

@bot.message_handler(commands=["set_price"])
def set_price(m):
    if m.from_user.id not in ADMIN_IDS: return
    args = m.text.split(" ", 2)
    if len(args)<3:
        bot.reply_to(m, "فرمت: /set_price عنوان_دوره مبلغ_تومان")
        return
    title = args[1].strip()
    try:
        price = int(args[2].strip())
    except:
        bot.reply_to(m, "مبلغ نامعتبر.")
        return
    conn = db(); cur = conn.cursor()
    cur.execute("UPDATE courses SET price=? WHERE title=?", (price, title))
    conn.commit(); conn.close()
    bot.reply_to(m, "قیمت بروزرسانی شد ✅")

# ---------------- Main Loop ----------------
def poll():
    print("Bot started.")
    bot.infinity_polling(timeout=60, long_polling_timeout=50)

if __name__ == "__main__":
    poll()
